// Julianisches Datum
// 16.12.2022 - 20.12.2022

// ****************************************************************************
// * Autor: Walter Fendt (www.walter-fendt.de)                                *
// * Dieses Programm darf - auch in vernderter Form - fr nicht-kommerzielle *
// * Zwecke verwendet und weitergegeben werden, solange dieser Hinweis nicht  *
// * entfernt wird.                                                           *
// **************************************************************************** 

// Sprachabhngige Texte sind einer eigenen Datei (zum Beispiel juliandate_de.js) abgespeichert.

// Attribute:

var rb1, rb2;                                              // Radiobuttons
var ch1, ch2;                                              // Auswahlfelder
var lbZone1, lbZone2;                                      // Ausgabefelder (Bezeichnungen der Zeitzone)
var ipDay, ipMonth, ipYear, ipHour, ipMinute;              // Ein-/Ausgabefelder (Datum und Uhrzeit)
var ipJD;                                                  // Ein-/Ausgabefeld (Julianisches Datum)
var bu;                                                    // OK-Button

var zone;                                                  // Zeitzone bezglich UTC (0, 1 oder 2)
var timeUTC;                                               // UTC-Zeit (Verbund)
var time;                                                  // Lokale Zeit (Verbund)
var jd;                                                    // Julianisches Datum

// Element der Schaltflche (aus HTML-Datei):
// id ..... ID im HTML-Befehl
// text ... Text (optional)

function getElement (id, text) {
  var e = document.getElementById(id);                     // Element
  if (text) e.innerHTML = text;                            // Text festlegen, falls definiert
  return e;                                                // Rckgabewert
  }
  
// Neues Auswahlfeld:
// id ... ID im HTML-Befehl
// t .... Array von Texten
// i .... Voreingestellter Index
  
function newSelect (id, t, i) {
  var ch = document.getElementById(id);                    // Neues Auswahlfeld
  for (var k=0; k<t.length; k++) {                         // Fr alle Indizes ...
    var o = document.createElement("option");              // Neues option-Element
    o.innerHTML = t[k];                                    // Text festlegen
    ch.add(o);                                             // option-Element hinzufgen
    }
  ch.selectedIndex = i;                                    // Startwert Index
  return ch;                                               // Rckgabewert
  }

// Start:

function start () {
  zone = 1;                                                // Zeitzone (UTC + 1 h)
  var d = new Date();                                      // Aktuelle Zeit
  timeUTC = {};                                            // Verbund fr UTC-Zeit (noch leer)
  timeUTC.year = d.getUTCFullYear();                       // Jahr (UTC)
  timeUTC.month = d.getUTCMonth()+1;                       // Monat (UTC)
  timeUTC.day = d.getUTCDate();                            // Tag im Monat (UTC)
  timeUTC.hour = d.getUTCHours();                          // Stunde (UTC)
  timeUTC.minute = d.getUTCMinutes();                      // Minute (UTC)
  time = getTime(timeUTC,zone);                            // Umrechnung in lokale Zeit  
  rb1 = getElement("rb1a");                                // Radiobutton (Julianisches Datum berechnen)
  rb1.checked = true;                                      // Radiobutton zunchst ausgewhlt
  getElement("rb1b",text01);                               // Erklrender Text (Julianisches Datum berechnen)
  rb2 = getElement("rb2a");                                // Radiobutton (Kalenderdatum berechnen)
  getElement("rb2b",text02);                               // Erklrender Text (Kalenderdatum berechnen)  
  getElement("ip3a",text03);                               // Erklrender Text (Zeitzone)
  ch1 = newSelect("ip3b",text04,zone);                     // Auswahlfeld (Zeitzone)
  lbZone1 = getElement("ip3c",text05[zone]);               // 1. Bezeichnung der Zeitzone
  lbZone2 = getElement("ip4c",text06[zone]);               // 2. Bezeichnung der Zeitzone
  getElement("ip5a",text07);                               // Erklrender Text (Datum)
  ipDay = getElement(english?"ip5d":"ip5b");               // Ein-/Ausgabefeld (Tag)
  getElement("ip5c",dateSeparator);                        // Trennzeichen Datum
  ipMonth = getElement(english?"ip5b":"ip5d");             // Ein-/Ausgabefeld (Monat)
  getElement("ip5e",dateSeparator);                        // Trennzeichen Datum
  ipYear = getElement("ip5f");                             // Ein-/Ausgabefeld (Jahr)
  var i = (time.year > 0 ? 1 : 0);                         // Index (vor/nach Christus)
  ch2 = newSelect("ip5g",text08,i);                        // Auswahlfeld (vor/nach Christus)
  getElement("ip6a",text09);                               // Erklrender Text (Uhrzeit) 
  ipHour = getElement("ip6b");                             // Ein-/Ausgabefeld (Stunde) 
  getElement("ip6c",unitHour);                             // Einheit (Stunde)
  ipMinute = getElement("ip6d");                           // Ein-/Ausgabefeld (Minute)
  getElement("ip6e",unitMinute);                           // Einheit (Minute)
  getElement("ip7a",text10);                               // Erklrender Text (Julianisches Datum)
  ipJD = getElement("ip7b");                               // Ein-/Ausgabefeld (Julianisches Datum)
  bu = getElement("bu",text11);                            // Schaltknopf (OK) 
  getElement("author",author);                             // Autor (und bersetzer) 
  
  updateInput();                                           // Ein-/Ausgabefelder fr Zeit aktualisieren
  reaction();                                              // Eingabe, Berechnung und Ausgabe
  
  rb1.onchange = reaction;                                 // Reaktion auf Radiobutton (Berechnung Julianisches Datum)
  rb2.onchange = reaction;                                 // Reaktion auf Radiobutton (Berechnung Kalenderdatum)
  ch1.onchange = reactionZone;                             // Reaktion auf Auswahlfeld (Zeitzone)
  ch2.onchange = reaction;                                 // Reaktion auf Auswahlfeld (vor/nach Christus)
  ipYear.onkeydown = reactionEnter;                        // Reaktion auf Enter-Taste (Eingabe Jahr)
  ipMonth.onkeydown = reactionEnter;                       // Reaktion auf Enter-Taste (Eingabe Monat)
  ipDay.onkeydown = reactionEnter;                         // Reaktion auf Enter-Taste (Eingabe Tag)
  ipHour.onkeydown = reactionEnter;                        // Reaktion auf Enter-Taste (Eingabe Stunde)
  ipMinute.onkeydown = reactionEnter;                      // Reaktion auf Enter-Taste (Eingabe Minute)
  ipJD.onkeydown = reactionEnter;                          // Reaktion auf Enter-Taste (Eingabe Julianisches Datum)
  bu.onclick = reaction;                                   // Reaktion auf Schaltknopf OK
  
  } // Ende der Methode start
  
// Reaktion auf Einstellung der Zeitzone:
// Seiteneffekt zone, time
// Wirkung auf Labels (Bezeichnungen der Zeitzone)
  
function reactionZone () {
  zone = ch1.selectedIndex;                                // Zeitzone (0 bis 2) aus Auswahlfeld
  time = getTime(timeUTC,zone);                            // Lokale Zeit berechnen                        
  updateInput();                                           // Eingabefelder aktualisieren
  lbZone1.innerHTML = text05[zone];                        // 1. Bezeichnung der Zeitzone
  lbZone2.innerHTML = text06[zone];                        // 2. Bezeichnung der Zeitzone
  }
  
// Eingabe von Datum und Uhrzeit, Berechnung, Ausgabe des julianischen Datums:
// Seiteneffekt time, timeUTC, jd
// Wirkung auf Ein-/Ausgabefelder
  
function reaction1 () {
  time.year = inputNumber(ipYear,0,true,1,10000);          // Jahr aus Eingabefeld
  if (ch2.selectedIndex == 0) time.year = -time.year;      // Jahr vor Christus
  time.month = inputNumber(ipMonth,0,true,1,12);           // Monat aus Eingabefeld
  var max = lengthMonth(time.year,time.month);             // Monatslnge
  time.day = inputNumber(ipDay,0,true,1,max);              // Tag aus Eingabefeld
  var e = (time.year == 1582 && time.month == 10);         // Flag fr Oktober 1582
  e = e && (time.day > 4 && time.day < 15);                // Flag fr bergangszeit 1582
  if (e) {                                                 // Falls bergangszeit ...
    alert(errorDate);                                      // Fehlermeldung
    time.day = 4;                                          // Rcksetzung auf 04.10.1582
    ipDay.value = "4";                                     // Eingabefeld aktualisieren
    }
  time.hour = inputNumber(ipHour,0,true,0,23);             // Stunde aus Eingabefeld
  time.minute = inputNumber(ipMinute,0,true,0,59);         // Minute aus Eingabefeld
  if (time.minute < 10) ipMinute.value = "0"+time.minute;  // Zweistellige Minutenangabe
  timeUTC = getUTC(time,zone);                             // Berechnung der UTC-Zeit
  jd = julianDateT(timeUTC);                               // Julianisches Datum
  ipJD.value = ToString(jd,3,true);                        // Ausgabe des julianischen Datums
  enable();                                                // Aktivierung bzw. Deaktivierung der Ein-/Ausgabefelder
  }
  
// Eingabe des julianischen Datums, Berechnung, Ausgabe von Datum und Uhrzeit:
// Seiteneffekt jd, timeUTC, time
// Wirkung auf Ein-/Ausgabefelder
  
function reaction2 () {
  jd = inputNumber(ipJD,3,true,-1931076,5373850);          // Julianisches Datum aus Eingabefeld        
  var u = dateJD(jd);                                      // UTC-Zeit (Verbund mit Attributen y, m,d)
  timeUTC.year = u.y;                                      // Jahr nach UTC-Zeit
  timeUTC.month = u.m;                                     // Monat nach UTC-Zeit
  timeUTC.day = Math.floor(u.d);                           // Tag nach UTC-Zeit
  var r = u.d-timeUTC.day;                                 // Restliche Zeit (Tage)
  if (r < 0) alert("Problem 1");
  if (r >= 1) alert("Problem 2");
  timeUTC.hour = Math.floor(24*r);                         // Volle Stunden nach UTC-Zeit
  r = 24*r-timeUTC.hour;                                   // Restliche Zeit (Stunden)
  if (r < 0) alert("Problem 3");
  if (r >= 24) alert("Problem 4");                                       
  timeUTC.minute = Math.round(60*r);                       // Minuten nach UTC-Zeit (gerundet)
  time = getTime(timeUTC,zone);                            // Berechnung der lokalen Zeit 
  ipYear.value = ToString(Math.abs(time.year),0,true);     // Ausgabe Jahr
  ipMonth.value = ToString(time.month,0,true);             // Ausgabe Monat
  ipDay.value = ToString(time.day,0,true);                 // Ausgabe Tag
  ipHour.value = ToString(time.hour,0,true);               // Ausgabe Stunde
  var sMin = ToString(time.minute,0,true);                 // Zeichenkette fr Minute (ein oder zwei Ziffern)
  ipMinute.value = (time.minute < 10 ? "0"+sMin : sMin);   // Ausgabe Minute
  ch2.selectedIndex = (time.year > 0 ? 1 : 0);             // Auswahlfeld vor/nach Christus
  enable();                                                // Aktivierung bzw. Deaktivierung der Ein-/Ausgabefelder
  }
  
// Eingabe, Berechnung, Ausgabe:
// Seiteneffekt time, timeUTC, jd
// Wirkung auf Ein-/Ausgabefelder
  
function reaction () {
  if (rb1.checked) reaction1();                            // Entweder Berechnung des julianischen Datums ...
  else reaction2();                                        // ... oder Berechnung von Kalenderdatum und Uhrzeit
  }
 
// Reaktion auf Tastendruck (nur auf Enter-Taste):
// Seiteneffekt time, timeUTC, jd 
  
function reactionEnter (e) {
  if (e.key && String(e.key) == "Enter"                    // Falls Entertaste (Firefox/Internet Explorer) ...
  || e.keyCode == 13)                                      // Falls Entertaste (Chrome) ...
    reaction();                                            // ... Daten bernehmen, rechnen, Ausgabe aktualisieren
  }

//-------------------------------------------------------------------------------------------------

// Berechnung der lokalen Zeit:
// u ... UTC-Zeit (Verbund mit den Attributen year, month, day, hour, minute)
// z ... Zeitzone (0, 1 oder 2)
// Rckgabewert: Lokale Zeit (Verbund mit den Attributen year, month, day, hour, minute)

function getTime (u, z) {
  if (z != 0 && z != 1 && z != 2) return undefined;        // Unerlaubter Wert von z
  var y = u.year;                                          // Jahr bernehmen
  var mo = u.month;                                        // Monat bernehmen
  var d = u.day;                                           // Tag bernehmen
  var h = u.hour+z;                                        // Stunde bernehmen und anpassen (eventuell ber 23)
  var mi = u.minute;                                       // Minute bernehmen
  if (h >= 24) {h -= 24; d++;}                             // Eventuell nchster Tag (eventuell ber Monatslnge)
  var lm = lengthMonth(y,mo);                              // Monatslnge
  if (d > lm) {d = 1; mo++;}                               // Eventuell nchster Monat (eventuell 13)
  if (mo > 12) {mo = 1; y++;}                              // Eventuell nchstes Jahr (eventuell 0)
  if (y == 0) y = 1;                                       // Jahr 0 durch 1 n. Chr. ersetzen
  return {year: y, month: mo, day: d, hour: h, minute: mi}; // Rckgabewert
  }
  
// Berechnung der UTC-Zeit:
// t ... Lokale Zeit (Verbund mit den Attributen year, month, day, hour, minute)
// z ... Zeitzone (0, 1 oder 2)
// Rckgabewert: UTC-Zeit (Verbund mit den Attributen year, month, day, hour, minute) 

function getUTC (t, z) {
  if (z != 0 && z != 1 && z != 2) return undefined;        // Unerlaubter Wert von z
  var y = t.year;                                          // Jahr bernehmen
  var mo = t.month;                                        // Monat bernehmen
  var d = t.day;                                           // Tag bernehmen
  var h = t.hour-z;                                        // Stunde bernehmen und anpassen (eventuell negativ)
  var mi = t.minute;                                       // Minute bernehmen
  if (h < 0) {h += 24; d--;}                               // Eventuell vorhergehender Tag (eventuell 0)
  var lm = lengthMonth(y,mo>1?mo-1:12);                    // Lnge des vorhergehenden Monats
  if (d < 1) {d = lm; mo--;}                               // Eventuell vorhergehender Monat (eventuell 0)
  if (mo < 1) {mo = 12; y--;}                              // Eventuell vorhergehendes Jahr (eventuell 0)
  if (y == 0) y = -1;                                      // Jahr 0 durch 1 v. Chr. ersetzen
  return {year: y, month: mo, day: d, hour: h, minute: mi}; // Rckgabewert
  }

// berprfung Schaltjahr:
// y ... Jahr nach astronomischer Zhlung
// Willkrliche Festlegung fr die Zeit vor Christus: Schalttag, falls Jahr nach astronomischer Zhlung
// durch 4 teilbar
  
function isLeapYear (y) {
  if (y < 1582) return (y%4 == 0);                         // Bis 1581 julianischer Kalender
  if (y%400 == 0) return true;                             // Teilbarkeit durch 400
  if (y%100 == 0) return false;                            // Teilbarkeit durch 100
  if (y%4 == 0) return true;                               // Teilbarkeit durch 4
  return false;                                            // Kein Schaltjahr
  }

// Monatslnge:
// y ... Jahr nach astronomischer Zhlung
// m ... Monat
    
function lengthMonth (y, m) {
  var ly = isLeapYear(y);                                  // Flag fr Schaltjahr
  if (m == 1) return 31;                                   // Januar
  if (m == 2) return (ly ? 29 : 28);                       // Februar
  if (m == 3) return 31;                                   // Mrz
  if (m == 4) return 30;                                   // April
  if (m == 5) return 31;                                   // Mai
  if (m == 6) return 30;                                   // Juni
  if (m == 7) return 31;                                   // Juli
  if (m == 8) return 31;                                   // August
  if (m == 9) return 30;                                   // September
  if (m == 10) return 31;                                  // Oktober
  if (m == 11) return 30;                                  // November
  if (m == 12) return 31;                                  // Dezember
  return undefined;                                        // Monat nicht im Bereich 1 bis 12
  }
 
// Umwandlung einer Zahl in eine Zeichenkette:
// n ..... Gegebene Zahl
// d ..... Zahl der Stellen
// fix ... Flag fr Nachkommastellen (im Gegensatz zu gltigen Ziffern)

function ToString (n, d, fix) {
  var s = (fix ? n.toFixed(d) : n.toPrecision(d));         // Zeichenkette mit Dezimalpunkt
  return s.replace(".",decimalSeparator);                  // Eventuell Punkt durch Komma ersetzen
  }
  
// Anpassung der Ein-/Ausgabefelder fr Datum und Uhrzeit:
// Wirkung auf die Eingabefelder ipDay, ipMonth, ipYear, ipHour, ipMinute und auf das Auswahlfeld vor/nach Christus
  
function updateInput () {
  ipDay.value = ToString(time.day,0,true);                 // Ein-/Ausgabefeld Tag
  ipMonth.value = ToString(time.month,0,true);             // Ein-/Ausgabefeld Monat
  ipYear.value = ToString(Math.abs(time.year),0,true);     // Ein-/Ausgabefeld Jahr
  ch2.selectedIndex = (time.year > 0 ? 1 : 0);             // Auswahlfeld vor/nach Christus 
  ipHour.value = ToString(time.hour,0,true);               // Ein-/Ausgabefeld Stunde
  ipMinute.value = ToString(time.minute,0,true);           // Ein-/Ausgabefeld Minute
  }
  
// Eingabe einer Zahl:
// ef .... Eingabefeld
// d ..... Zahl der Stellen
// fix ... Flag fr Nachkommastellen (im Gegensatz zu gltigen Ziffern)
// min ... Minimum des erlaubten Bereichs
// max ... Maximum des erlaubten Bereichs
// Rckgabewert: Zahl oder NaN
// Wirkung auf Eingabefeld
  
function inputNumber (ef, d, fix, min, max) {
  var s = ef.value;                                        // Zeichenkette im Eingabefeld
  s = s.replace(",",".");                                  // Eventuell Komma in Punkt umwandeln
  var n = Number(s);                                       // Umwandlung in Zahl, falls mglich
  if (isNaN(n)) n = 0;                                     // Sinnlose Eingaben als 0 interpretieren 
  if (n < min) n = min;                                    // Falls Zahl zu klein, korrigieren
  if (n > max) n = max;                                    // Falls Zahl zu gro, korrigieren
  ef.value = ToString(n,d,fix);                            // Eingabefeld eventuell korrigieren
  return n;                                                // Rckgabewert
  }
  
// Aktivierung bzw. Deaktivierung der Eingabefelder:
  
function enable () {
  ipYear.readOnly = rb2.checked;                           // Ein-/Ausgabefeld Jahr
  ipMonth.readOnly = rb2.checked;                          // Ein-/Ausgabefeld Monat
  ipDay.readOnly = rb2.checked;                            // Ein-/Ausgabefeld Tag
  ipHour.readOnly = rb2.checked;                           // Ein-/Ausgabefeld Stunde
  ipMinute.readOnly = rb2.checked;                         // Ein-/Ausgabefeld Minute
  ipJD.readOnly = rb1.checked;                             // Ein-/Ausgabefeld Julianisches Datum
  }
  
// berprfung, ob Datum korrekt:
// y ... Jahr (normale Zhlung ohne Jahr 0)
// m ... Monat
// d ... Tag (einschlielich Bruchteil)
  
function isCorrectDate (y, m, d) {
  if (y == 0) return false;                                // Rckgabewert fr das Jahr 0
  if (y < 0) y++;                                          // Astronomische Zhlung
  if (m < 1 || m > 12) return false;                       // Fehler beim Monat
  var dd = Math.floor(d);                                  // Ganzer Tag
  if (dd < 1 || dd > lengthMonth(y,m)) return false;       // Fehler beim Tag
  if (y == 1582 && m == 10 && d > 4 && d < 15) return false; // Fehler wegen bergang julianisch/gregorianisch
  return true;                                             // Korrektes Datum
  }
  
// Julianisches Datum (fr 0 Uhr UTC):
// year .... Jahr (normale Zhlung ohne Jahr 0)
// month ... Monat (1 bis 12)
// day ..... Tag (einschlielich Bruchteil)
// Algorithmus nach Oliver Montenbruck, Grundlagen der Ephemeridenrechnung
  
function julianDate (year, month, day) {
  var c = isCorrectDate(year,month,day);                   // berprfung, ob Datum korrekt
  if (!c) return undefined;                                // Rckgabewert fr das Jahr 0
  if (year < 0) year++;                                    // Astronomische Zhlung
  var y = year, m = month;                                 // Jahr und Monat
  if (m <= 2) {y--; m += 12;}                              // Falls Januar oder Februar, Umnummerierung
  var greg = false;                                        // Flag fr gregorianischen Kalender, Startwert
  if (y > 1582) greg = true;                               // Nach 1582 Flag setzen
  else if (y == 1582 && m > 10) greg = true;               // Fr November/Dezember 1582 Flag setzen
  else if (y == 1582 && m == 10 && day >= 15) greg = true; // Fr 15. bis 31. Oktober 1582 Flag setzen
  var b = (greg ? Math.floor(y/400)-Math.floor(y/100) : -2); // Hilfsgre
  var jd = Math.floor(365.25*y);                           // Summand fr Jahr
  jd += Math.floor(30.6001*(m+1));                         // Summand fr Monat
  jd += b+1720996.5+day;                                   // Weitere Summanden
  return jd;                                               // Rckgabewert
  }
  
// Julianisches Datum:
// u ... UTC-Zeit (Verbund mit den Attributen year, month, day, hour, minute)
  
function julianDateT (u) {
  var y = u.year;                                          // Jahr
  var mo = u.month;                                        // Monat
  var d = u.day;                                           // Tag
  var h = u.hour;                                          // Stunde
  var mi = u.minute;                                       // Minute
  return julianDate(y,mo,d)+h/24+mi/1440;                  // Rckgabewert (Algorithmus Montenbruck)
  }
    
// Umrechnung vom julianischen Datum in ein normales Datum:
// j ... Gegebenes julianisches Datum
// Rckgabewert: Verbund, bestehend aus Jahr y (normale Zhlung), Monat m und Tag d (im Allgemeinen nicht ganzzahlig)
// Algorithmus nach Oliver Montenbruck, Grundlagen der Ephemeridenrechnung
    
function dateJD (j) {
  var a = Math.floor(j+0.5);                               // Hilfsgre a
  var c = a+1524;                                          // Hilfsgre c (julianischer Kalender)
  if (a >= 2299161) {                                      // Falls gregorianischer Kalender ...
    var b = Math.floor((a-1867216.25)/36524.25);           // Lokale Hilfsgre
    c = a+b-Math.floor(b/4)+1525;                          // Hilfsgre c (gregorianischer Kalender)
    }
  var d = Math.floor((c-122.1)/365.25);                    // Hilfsgre d
  var e = Math.floor(365.25*d);                            // Hilfsgre e
  var f = Math.floor((c-e)/30.6001);                       // Hilfsgre f
  var day = c-e-Math.floor(30.6001*f)+(j+0.5-a);           // Tag (im Allgemeinen nicht ganzzahlig)
  var month = f-1-12*Math.floor(f/14);                     // Monat (ganzzahlig)
  var year = d-4715-Math.floor((7+month)/10);              // Jahr (ganzzahlig, astronomische Zhlung)
  if (year <= 0) year--;                                   // Jahr (normale Zhlung ohne Jahr 0)
  return {y: year, m: month, d: day};                      // Rckgabewert
  }
        
document.addEventListener("DOMContentLoaded",start,false); // Nach dem Laden der Seite Start-Methode aufrufen

