// Online-Rechner Sternzeit
// 21.12.2022 - 22.12.2022

// ****************************************************************************
// * Autor: Walter Fendt (www.walter-fendt.de)                                *
// * Dieses Programm darf - auch in vernderter Form - fr nicht-kommerzielle *
// * Zwecke verwendet und weitergegeben werden, solange dieser Hinweis nicht  *
// * entfernt wird.                                                           *
// **************************************************************************** 

// Sprachabhngige Texte sind einer eigenen Datei (zum Beispiel siderealtime_de.js) abgespeichert.

// Konstanten:

var D_DAY = 6;                                             // Tage mit 6 Nachkommastellen
var D_HOUR = 5;                                            // Stunden mit 5 Nachkommastellen
var D_SEC = 1;                                             // Zeitsekunden mit 1 Nachkommastelle
var D_DEG = 4;                                             // Winkelgrad mit 4 Nachkommastellen
var D_ARCSEC = 0;                                          // Winkelsekunden ohne Nachkommastellen

// Attribute:

var ipDay, ipMonth, ipYear;                                // Eingabefelder (Datum)
var ipHour, ipMinute, ipSecond;                            // Eingabefelder (Uhrzeit)
var ipLongDeg, ipLongMin, ipLongSec;                       // Eingabefelder (geographische Lnge)
var chLong;                                                // Auswahlfeld (stliche/westliche Lnge)
var opJD;                                                  // Ausgabefeld (Julianisches Datum)
var opGMST1, opGMST2;                                      // Ausgabefelder (mittlere Greenwich-Sternzeit, Zeitma/Gradma)
var opLMST1, opLMST2;                                      // Ausgabefelder (mittlere Ortssternzeit, Zeitma/Gradma)
var bu;                                                    // OK-Button
var timeUT;                                                // UT-Zeit (Verbund)
var long;                                                  // Betrag der geographischen Lnge (Verbund)
var jd;                                                    // Julianisches Datum

// Element der Schaltflche (aus HTML-Datei):
// id ..... ID im HTML-Befehl
// text ... Text (optional)

function getElement (id, text) {
  var e = document.getElementById(id);                     // Element
  if (text) e.innerHTML = text;                            // Text festlegen, falls definiert
  return e;                                                // Rckgabewert
  }
  
// Neues Auswahlfeld:
// id ... ID im HTML-Befehl
// t .... Array von Texten
// i .... Voreingestellter Index
  
function newSelect (id, t, i) {
  var ch = document.getElementById(id);                    // Neues Auswahlfeld
  for (var k=0; k<t.length; k++) {                         // Fr alle Indizes ...
    var o = document.createElement("option");              // Neues option-Element
    o.innerHTML = t[k];                                    // Text festlegen
    ch.add(o);                                             // option-Element hinzufgen
    }
  ch.selectedIndex = i;                                    // Startwert Index
  return ch;                                               // Rckgabewert
  }
  
// Aktivierung bzw. Deaktivierung der Ein- und Ausgabefelder:
  
function enable () {
  ipYear.readOnly = false;                                 // Eingabefeld Jahr
  ipMonth.readOnly = false;                                // Eingabefeld Monat
  ipDay.readOnly = false;                                  // Eingabefeld Tag
  ipHour.readOnly = false;                                 // Eingabefeld Stunde
  ipMinute.readOnly = false;                               // Eingabefeld Minute
  ipSecond.readOnly = false;                               // Eingabefeld Sekunde
  ipLongDeg.readOnly = false;                              // Eingabefeld geogr. Lnge, Grad
  ipLongMin.readOnly = false;                              // Eingabefeld geogr. Lnge, Winkelminuten
  ipLongSec.readOnly = false;                              // Eingabefeld geogr. Lnge, Winkelsekunden
  opJD.readOnly = true;                                    // Ausgabefeld Julianisches Datum
  opGMST1.readOnly = true;                                 // Ausgabefeld Mittlere Greenwich-Sternzeit, Zeitma
  opGMST2.readOnly = true;                                 // Ausgabefeld Mittlere Greenwich-Sternzeit, Gradma
  opLMST1.readOnly = true;                                 // Ausgabefeld Mittlere Ortssternzeit, Zeitma
  opLMST2.readOnly = true;                                 // Ausgabefeld Mittlere Ortssternzeit, Gradma
  }

// Start:

function start () {
  var d = new Date();                                      // Aktuelle Zeit
  timeUT = {                                               // Verbund fr UT-Zeit
    year: d.getUTCFullYear(),                              // Jahr (UTC)
    month: d.getUTCMonth()+1,                              // Monat (UTC) 
    day: d.getUTCDate(),                                   // Tag (UTC)
    hour: d.getUTCHours(),                                 // Stunde (UTC)
    minute: d.getUTCMinutes(),                             // Minute (UTC)
    second: d.getUTCSeconds()                              // Sekunde (UTC)
    };                                             
  long = {deg: 10, min: 0, sec: 0};                        // Verbund fr geographische Lnge
  getElement("ip1a",text01);                               // Erklrender Text (Datum)
  ipDay = getElement(english?"ip1d":"ip1b");               // Eingabefeld (Tag)
  getElement("ip1c",dateSeparator);                        // Trennzeichen Datum
  ipMonth = getElement(english?"ip1b":"ip1d");             // Eingabefeld (Monat)
  getElement("ip1e",dateSeparator);                        // Trennzeichen Datum
  ipYear = getElement("ip1f");                             // Eingabefeld (Jahr)
  getElement("ip2a",text02);                               // Erklrender Text (Uhrzeit) 
  ipHour = getElement("ip2b");                             // Eingabefeld (Stunde) 
  getElement("ip2c",unitHour);                             // Einheit (Stunde)
  ipMinute = getElement("ip2d");                           // Eingabefeld (Minute)
  getElement("ip2e",unitMinute);                           // Einheit (Minute)
  ipSecond = getElement("ip2f");                           // Eingabefeld (Sekunde)
  getElement("ip2g",unitSecond);                           // Einheit (Sekunde)
  getElement("ip3a",text03);                               // Erklrender Text (geogr. Lnge)
  ipLongDeg = getElement("ip3b");                          // Eingabefeld (geogr. Lnge, Winkelgrad)
  getElement("ip3c",unitDeg);                              // Einheit (Winkelgrad)
  ipLongMin = getElement("ip3d");                          // Eingabefeld (geogr. Lnge, Winkelminuten)
  getElement("ip3e",unitArcMin);                           // Einheit (Winkelminuten)
  ipLongSec = getElement("ip3f");                          // Eingabefeld (geogr. Lnge, Winkelsekunden)
  getElement("ip3g",unitArcSec);                           // Einheit (Winkelsekunden) 
  chLong = newSelect("ip3h",text08,0);                     // Auswahlfeld (stliche/westliche Lnge)
  getElement("op4a",text04);                               // Erklrender Text (Julianisches Datum)
  opJD = getElement("op4b");                               // Ausgabefeld (Julianisches Datum)
  getElement("op5a",text05);                               // Erklrender Text (mittlere Greenwich-Sternzeit)
  opGMST1 = getElement("op5b");                            // Ausgabefeld (mittlere Grennwich-Sternzeit, Zeitma)
  opGMST2 = getElement("op5c");                            // Ausgabefeld (mittlere Greenwich-Sternzeit, Gradma)
  getElement("op6a",text06);                               // Erklrender Text (mittlere Ortssternzeit)
  opLMST1 = getElement("op6b");                            // Ausgabefeld (mittlere Ortssternzeit, Zeitma)
  opLMST2 = getElement("op6c");                            // Ausgabefeld (mittlere Ortssternzeit, Gradma)  
  bu = getElement("bu",text07);                            // Schaltknopf (OK) 
  getElement("author",author);                             // Autor (und bersetzer) 
  enable();                                                // Aktivierung bzw. Deaktivierung der Ein- und Ausgabefelder
  
  updateInput();                                           // Eingabefelder aktualisieren
  reaction();                                              // Eingabe, Berechnung und Ausgabe
  
  ipYear.onkeydown = reactionEnter;                        // Reaktion auf Enter-Taste (Eingabe Jahr)
  ipMonth.onkeydown = reactionEnter;                       // Reaktion auf Enter-Taste (Eingabe Monat)
  ipDay.onkeydown = reactionEnter;                         // Reaktion auf Enter-Taste (Eingabe Tag)
  ipHour.onkeydown = reactionEnter;                        // Reaktion auf Enter-Taste (Eingabe Stunde)
  ipMinute.onkeydown = reactionEnter;                      // Reaktion auf Enter-Taste (Eingabe Minute)
  ipSecond.onkeydown = reactionEnter;                      // Reaktion auf Enter-Taste (Eingabe Sekunde)
  ipLongDeg.onkeydown = reactionEnter;                     // Reaktion auf Enter-Taste (Eingabe geogr. Lnge, Grad)
  ipLongMin.onkeydown = reactionEnter;                     // Reaktion auf Enter-Taste (Eingabe geogr. Lnge, Winkelminuten)
  ipLongSec.onkeydown = reactionEnter;                     // Reaktion auf Enter-Taste (Eingabe geogr. Lnge, Winkelsekunden)
  chLong.onchange = reaction;                              // Reaktion auf Auswahlfeld (stliche/westliche Lnge)
  bu.onclick = reaction;                                   // Reaktion auf Schaltknopf OK
  
  } // Ende der Methode start
  
// Eingabe von Datum und Uhrzeit, Berechnung, Ausgabe:
// Seiteneffekt timeUT, jd
// Wirkung auf Ein- und Ausgabefelder
  
function reaction () {
  timeUT.year = inputNumber(ipYear,0,true,1900,2100);      // Jahr aus Eingabefeld
  timeUT.month = inputNumber(ipMonth,0,true,1,12);         // Monat aus Eingabefeld
  var max = lengthMonth(timeUT.year,timeUT.month);         // Monatslnge
  timeUT.day = inputNumber(ipDay,0,true,1,max);            // Tag aus Eingabefeld
  timeUT.hour = inputNumber(ipHour,0,true,0,23);           // Stunde aus Eingabefeld
  timeUT.minute = inputNumber(ipMinute,0,true,0,59);       // Minute aus Eingabefeld
  if (timeUT.minute < 10)                                  // Falls Minutenzahl einstellig ... 
    ipMinute.value = "0"+ipMinute.value;                   // Zweistellige Angabe
  timeUT.second = inputNumber(ipSecond,D_SEC,true,0,59.9); // Sekunde aus Eingabefeld
  if (timeUT.second < 10)                                  // Falls Sekundenzahl einstellig ... 
    ipSecond.value = "0"+ipSecond.value;                   // Zweistellige Angabe  
  jd = julianDateT(timeUT);                                // Julianisches Datum
  opJD.value = ToString(jd,D_DAY,true);                    // Ausgabe des julianischen Datums
  long.deg = inputNumber(ipLongDeg,0,true,0,180);          // Grad Lnge aus Eingabefeld
  long.min = inputNumber(ipLongMin,0,true,0,59);           // Winkelminute Lnge aus Eingabefeld
  long.sec = inputNumber(ipLongSec,D_ARCSEC,true,0,59);    // Winkelsekunde Lnge aus Eingabefeld
  if (long.deg == 180) {                                   // Falls 180 eingetragen ...
    long.min = long.sec = 0;                               // Winkelminuten und -sekunden gleich 0
    ipLongMin.value = "0";                                 // Eingabefeld fr Winkelminuten aktualisieren
    ipLongSec.value = ToString(0,D_ARCSEC,true);           // Eingabefeld fr Winkelsekunden aktualisieren
    }
  var t = gmst(timeUT);                                    // Mittlere Greenwich-Sternzeit (Sekunden)
  opGMST1.value = toHourMinSec(t)+" = "+toHours(t);        // Ausgabe im Zeitma
  opGMST2.value = toDegMinSec(t)+" = "+toDegrees(t);       // Ausgabe im Gradma
  t = lmst(timeUT);                                        // Mittlere Ortssternzeit (Sekunden)
  opLMST1.value = toHourMinSec(t)+" = "+toHours(t);        // Ausgabe im Zeitma
  opLMST2.value = toDegMinSec(t)+" = "+toDegrees(t);       // Ausgabe im Gradma  
  }  
 
// Reaktion auf Tastendruck (nur auf Enter-Taste):
// Seiteneffekt timeUT, jd 
  
function reactionEnter (e) {
  if (e.key && String(e.key) == "Enter"                    // Falls Entertaste (Firefox/Internet Explorer) ...
  || e.keyCode == 13)                                      // Falls Entertaste (Chrome) ...
    reaction();                                            // ... Daten bernehmen, rechnen, Ausgabe aktualisieren
  }

//-------------------------------------------------------------------------------------------------

// berprfung Schaltjahr:
// y ... Jahr nach astronomischer Zhlung
// Willkrliche Festlegung fr die Zeit vor Christus: Schalttag, falls Jahr nach astronomischer Zhlung
// durch 4 teilbar
  
function isLeapYear (y) {
  if (y < 1582) return (y%4 == 0);                         // Bis 1581 julianischer Kalender
  if (y%400 == 0) return true;                             // Teilbarkeit durch 400
  if (y%100 == 0) return false;                            // Teilbarkeit durch 100
  if (y%4 == 0) return true;                               // Teilbarkeit durch 4
  return false;                                            // Kein Schaltjahr
  }

// Monatslnge:
// y ... Jahr nach astronomischer Zhlung
// m ... Monat
    
function lengthMonth (y, m) {
  var ly = isLeapYear(y);                                  // Flag fr Schaltjahr
  if (m == 1) return 31;                                   // Januar
  if (m == 2) return (ly ? 29 : 28);                       // Februar
  if (m == 3) return 31;                                   // Mrz
  if (m == 4) return 30;                                   // April
  if (m == 5) return 31;                                   // Mai
  if (m == 6) return 30;                                   // Juni
  if (m == 7) return 31;                                   // Juli
  if (m == 8) return 31;                                   // August
  if (m == 9) return 30;                                   // September
  if (m == 10) return 31;                                  // Oktober
  if (m == 11) return 30;                                  // November
  if (m == 12) return 31;                                  // Dezember
  return undefined;                                        // Monat nicht im Bereich 1 bis 12
  }
 
// Umwandlung einer Zahl in eine Zeichenkette:
// n ..... Gegebene Zahl
// d ..... Zahl der Stellen (Konstanten fr Genauigkeit verwenden!)
// fix ... Flag fr Nachkommastellen (im Gegensatz zu gltigen Ziffern)

function ToString (n, d, fix) {
  var s = (fix ? n.toFixed(d) : n.toPrecision(d));         // Zeichenkette mit Dezimalpunkt
  return s.replace(".",decimalSeparator);                  // Eventuell Punkt durch Komma ersetzen
  }
  
// Anpassung der Eingabefelder fr Datum und Uhrzeit:
// Wirkung auf die Eingabefelder ipDay, ipMonth, ipYear, ipHour, ipMinute, ipSecond, ipLongDeg, ipLongMin, ipLongSec
  
function updateInput () {
  ipDay.value = ToString(timeUT.day,0,true);               // Eingabefeld Tag
  ipMonth.value = ToString(timeUT.month,0,true);           // Eingabefeld Monat
  ipYear.value = ToString(Math.abs(timeUT.year),0,true);   // Eingabefeld Jahr 
  ipHour.value = ToString(timeUT.hour,0,true);             // Eingabefeld Stunde
  ipMinute.value = ToString(timeUT.minute,0,true);         // Eingabefeld Minute
  ipSecond.value = ToString(timeUT.second,D_SEC,true);     // Eingabefeld Sekunde
  ipLongDeg.value = ToString(long.deg,0,true);             // Eingabefeld geogr. Lnge (Grad)
  ipLongMin.value = ToString(long.min,0,true);             // Eingabefeld geogr. Lnge (Winkelminuten)
  ipLongSec.value = ToString(long.sec,D_ARCSEC,true);      // Eingabefeld geogr. Lnge (Winkelsekunden)
  }
  
// Eingabe einer Zahl:
// ef .... Eingabefeld
// d ..... Zahl der Stellen
// fix ... Flag fr Nachkommastellen (im Gegensatz zu gltigen Ziffern)
// min ... Minimum des erlaubten Bereichs
// max ... Maximum des erlaubten Bereichs
// Rckgabewert: Zahl oder NaN
// Wirkung auf Eingabefeld
  
function inputNumber (ef, d, fix, min, max) {
  var s = ef.value;                                        // Zeichenkette im Eingabefeld
  s = s.replace(",",".");                                  // Eventuell Komma in Punkt umwandeln
  var n = Number(s);                                       // Umwandlung in Zahl, falls mglich
  if (isNaN(n)) n = 0;                                     // Sinnlose Eingaben als 0 interpretieren 
  if (n < min) n = min;                                    // Falls Zahl zu klein, korrigieren
  if (n > max) n = max;                                    // Falls Zahl zu gro, korrigieren
  ef.value = ToString(n,d,fix);                            // Eingabefeld eventuell korrigieren
  return n;                                                // Rckgabewert
  }
  
// berprfung, ob Datum korrekt:
// y ... Jahr (normale Zhlung ohne Jahr 0)
// m ... Monat
// d ... Tag (einschlielich Bruchteil)
  
function isCorrectDate (y, m, d) {
  if (y == 0) return false;                                // Rckgabewert fr das Jahr 0
  if (y < 0) y++;                                          // Astronomische Zhlung
  if (m < 1 || m > 12) return false;                       // Fehler beim Monat
  var dd = Math.floor(d);                                  // Ganzer Tag
  if (dd < 1 || dd > lengthMonth(y,m)) return false;       // Fehler beim Tag
  if (y == 1582 && m == 10 && d > 4 && d < 15)             // Falls bergangszeit im Oktober 1582 ... 
    return false;                                          // Fehler
  return true;                                             // Korrektes Datum
  }
  
// Julianisches Datum (fr 0 Uhr UT):
// year .... Jahr (normale Zhlung ohne Jahr 0)
// month ... Monat (1 bis 12)
// day ..... Tag (einschlielich Bruchteil)
// Algorithmus nach Oliver Montenbruck, Grundlagen der Ephemeridenrechnung
  
function julianDate (year, month, day) {
  var c = isCorrectDate(year,month,day);                   // berprfung, ob Datum korrekt
  if (!c) return undefined;                                // Rckgabewert fr das Jahr 0
  if (year < 0) year++;                                    // Astronomische Zhlung
  var y = year, m = month;                                 // Jahr und Monat
  if (m <= 2) {y--; m += 12;}                              // Falls Januar oder Februar, Umnummerierung
  var greg = false;                                        // Flag fr gregorianischen Kalender, Startwert
  if (y > 1582) greg = true;                               // Nach 1582 Flag setzen
  else if (y == 1582 && m > 10) greg = true;               // Fr November/Dezember 1582 Flag setzen
  else if (y == 1582 && m == 10 && day >= 15) greg = true; // Fr 15. bis 31. Oktober 1582 Flag setzen
  var b = (greg ? Math.floor(y/400)-Math.floor(y/100) : -2); // Hilfsgre
  var jd = Math.floor(365.25*y);                           // Summand fr Jahr
  jd += Math.floor(30.6001*(m+1));                         // Summand fr Monat
  jd += b+1720996.5+day;                                   // Weitere Summanden
  return jd;                                               // Rckgabewert
  }
  
// Julianisches Datum:
// u ... UT-Zeit (Verbund mit den Attributen year, month, day, hour, minute, second)
  
function julianDateT (u) {
  var y = u.year;                                          // Jahr
  var mo = u.month;                                        // Monat
  var d = u.day;                                           // Tag (ganzzahlig)
  var h = u.hour;                                          // Stunde
  var mi = u.minute;                                       // Minute
  var se = u.second;                                       // Sekunde
  return julianDate(y,mo,d)+h/24+mi/1440+se/86400;         // Rckgabewert (Algorithmus Montenbruck)
  }
  
// Mittlere Greenwich-Sternzeit:
// u ... UT-Zeit (Verbund mit den Attributen year, month, day, hour, minute, second)
// Rckgabewert: Mittlere Greenwich-Sternzeit in Sekunden, Bereich 0 bis 86400

function gmst (u) {
  var jd = julianDate(u.year,u.month,u.day);               // Julianisches Datum fr 0 Uhr UT
  var t = (jd-2451545)/36525;                              // Jahrhunderte seit Standardepoche J2000.0
  var s0 = ((-0.0000062*t+0.093104)*t+8640184.812866)*t+24110.54841; // Sternzeit fr 0 Uhr UT (Sekunden)
  var n = Math.floor(s0/86400);                            // Anzahl der ganzen Tage
  s0 -= n*86400;                                           // Ganze Tage subtrahieren
  s0 += (u.hour*3600+u.minute*60+u.second)*1.00273790935;  // Bercksichtigung der Uhrzeit (UT)
  if (s0 > 86400) s0 -= 86400;                             // Eventuell einen Tag subtrahieren
  return s0;                                               // Rckgabewert
  }
  
// Mittlere Ortssternzeit:
// u ... UT-Zeit (Verbund mit den Attributen year, month, day, hour, minute, second)
// Rckgabewert: Mittlere Ortssternzeit in Sekunden, Bereich 0 bis 86400

function lmst (u) {
  var lon = long.deg*240+long.min*4+long.sec/15;           // Betrag der geographischen Lnge (Zeitsekunden)
  if (chLong.selectedIndex == 1) lon = -lon;               // Falls westliche Lnge, Vorzeichenwechsel
  var t = gmst(u)+lon;                                     // Sternzeit (Zeitsekunden)
  if (t > 86400) t -= 86400;                               // Eventuell einen Tag subtrahieren
  if (t < 0) t += 86400;                                   // Eventuell einen Tag addieren
  return t;                                                // Rckgabewert
  }
 
// Zeichenkette fr Zeitangabe (Stunden, Minuten, Sekunden): 
// t ... Zeit (Sekunden, Bereich 0 bis 86400)
  
function toHourMinSec (t) {
  var n = Math.floor(t/3600);                              // Zahl der ganzen Stunden
  var s = String(n)+" "+unitHour+" ";                      // Anfang der Zeichenkette (Stunden)
  t -= n*3600;                                             // Restliche Zeit (Sekunden)
  n = Math.floor(t/60);                                    // Zahl der ganzen Minuten
  s += String(n)+" "+unitMinute+" ";                       // Zeichenkette ergnzen (Minuten)
  t -= n*60;                                               // Restliche Zeit (Sekunden)
  s += ToString(t,D_SEC,true)+" "+unitSecond;              // Zeichenkette ergnzen (Sekunden)
  return s;                                                // Rckgabewert
  }
  
// Zeichenkette fr Zeitangabe (Stunden):
// t ... Zeit (Sekunden, Bereich 0 bis 86400)
  
function toHours (t) {
  return ToString(t/3600,D_HOUR,true)+" "+unitHour;        // Rckgabewert
  }
  
// Zeichenkette fr Winkelangabe (Grad, Winkelminuten, Winkelsekunden):
// t ... Zeit (Sekunden, Bereich 0 bis 86400)
  
function toDegMinSec (t) {
  var n = Math.floor(t/240);                               // Zahl der ganzen Winkelgrade (240 s entspricht 1)
  var s = String(n)+unitDeg+" ";                           // Anfang der Zeichenkette (Winkelgrade)
  t -= n*240;                                              // Restliche Zeit (Sekunden)
  n = Math.floor(t/4);                                     // Zahl der ganzen Winkelminuten (4 s entspricht 1')
  s += String(n)+unitArcMin+" ";                           // Zeichenkette ergnzen (Winkelminuten)
  t -= n*4;                                                // Restliche Zeit (Sekunden)
  s += ToString(t*15,D_ARCSEC,true)+unitArcSec;            // Zeichenkette ergnzen (Winkelsekunden)                        
  return s;                                                // Rckgabewert
  }
  
// Zeichenkette fr Winkelangabe (Grad):
// t ... Zeit (Sekunden, Bereich 0 bis 86400)
  
function toDegrees (t) {
  return ToString(t/240,D_DEG,true)+unitDeg;               // Rckgabewert 
  }
           
document.addEventListener("DOMContentLoaded",start,false); // Nach dem Laden der Seite Start-Methode aufrufen

