// Klasse Monomial (Monom mit Variablen a bis z, ganzzahliger Koeffizient)
// Der Koeffizient wird als BigInt-Objekt mit dem Bezeichner coeff gespeichert. Nullmonome (mit coeff == 0n) sind erlaubt. 
// Fr die Exponenten der vorkommenden Variablen wird ein Array mit dem Bezeichner expo angelegt; dieses Array
// enthlt normale, nicht negative Zahlen (keine BigInt-Objekte!).
// 01.07.2020 - 16.08.2020

class Monomial {

// Konstruktor fr ein Monom mit hchstens einer Variablen:
// c ... Zeichenkette fr Koeffizient (ganzzahlig, Defaultwert 1)
// v ... Zeichenkette fr Variable ('a' bis 'z', optional)
// e ... Zeichenkette fr Exponent (natrliche Zahl oder 0, optinal, Defaultwert 1)

  constructor (c, v, e) {
    this.coeff = (c!=undefined ? BigInt(c) : 1n);          // Koeffizient bernehmen oder Koeffizient 1
    var n = variables.length;                              // Zahl der vorkommenden Variablen (hchstens 26)
    this.expo = new Array(n);                              // Neues Array fr Exponenten
    for (var i=0; i<n; i++) this.expo[i] = 0;              // Exponenten zunchst gleich 0
    if (v == undefined) return;                            // Falls Variable undefiniert, abbrechen (Monom ohne Variable)
    i = variables.indexOf(v);                              // Index der Variablen v (0 bis variables.length-1)
    this.expo[i] = (e ? Number(e) : 1);                    // Exponent bernehmen oder gleich 1 setzen
    }
    
// Kopie:

  clone () {
    var r = new Monomial();                                // Neues Monom
    r.coeff = this.coeff;                                  // Koeffizient bernehmen
    for (var i=0; i<variables.length; i++)                 // Fr alle Variablen-Indizes ...
      r.expo[i] = this.expo[i];                            // Exponent bernehmen
    return r;                                              // Rckgabewert
    }
    
// Vorzeichenumkehr: Das Monom wird dabei gendert.

  changeSign () {
    this.coeff = 0n-this.coeff;                            // Umgekehrtes Vorzeichen beim Koeffizienten
    }
    
// Umgekehrtes Monom:

  negate () {
    var r = this.clone();                                  // Kopie des gegebenen Monoms
    r.coeff = 0n-r.coeff;                                  // Vorzeichen des Koeffizienten umkehren
    return r;                                              // Rckgabewert
    }
    
// berprfung, ob Monom gleich 0:

  isZero () {return (this.coeff == 0n);}                   
  
// berprfung, ob ganze Zahl:

  isInt () {
    if (this.isZero()) return true;                        // Rckgabewert, falls Nullmonom
    for (var i=0; i<variables.length; i++)                 // Fr alle Variablen-Indizes ...
      if (this.expo[i] != 0) return false;                 // Rckgabewert, falls Exponent ungleich 0
    return true;                                           // Rckgabewert, falls alle Exponenten gleich 0
    }
    
// berprfung, ob Monom gleich 1:

  isOne () {
    return (this.isInt() && this.coeff == 1n);             // Rckgabewert 
    }
    
// Summe des gegebenen Monoms und eines weiteren Monoms:
// m ... Zweites Monom
// Bei nicht gleichartigen Monomen Rckgabewert undefined

  add (m) {
    var r = new Monomial();                                // Neues Monom (fr Summe)
    for (var i=0; i<variables.length; i++) {               // Fr alle Variablen-Indizes ...
      var e = this.expo[i];                                // Aktueller Exponent
      r.expo[i] = e;                                       // Exponent vom gegebenen Monom bernehmen
      if (m.expo[i] != e) return undefined;                // Falls abweichende Exponenten, Rckgabewert undefined
      }
    r.coeff = this.coeff+m.coeff;                          // Koeffizienten addieren
    return r;                                              // Rckgabewert
    }
    
// berprfung der Gleichartigkeit:
// m ... Vergleichsmonom

  isEquiv (m) {
    if (this.coeff == 0n) return true;                     // Rckgabewert, falls gegebenes Monom gleich 0
    if (m.coeff == 0n) return true;                        // Rckgabewert, falls Vergleichsmonom gleich 0
    for (var i=0; i<variables.length; i++)                 // Fr alle Variablen-Indizes ...
      if (this.expo[i] != m.expo[i]) return false;         // Rckgabewert, falls abweichender Exponent
    return true;                                           // Rckgabewert, falls alle Exponenten gleich                                         
    }
    
// Produkt des gegebenen Monoms und eines weiteren Monoms:
// m ... Zweites Monom

  mul (m) {
    var r = new Monomial();                                // Neues Monom (fr Produkt)
    r.coeff = this.coeff*m.coeff;                          // Koeffizient (Produkt der gegebenen Koeffizienten)
    for (var i=0; i<variables.length; i++)                 // Fr alle Variablen-Indizes ...
      r.expo[i] = this.expo[i]+m.expo[i];                  // Exponent (Summe der gegebenen Exponenten)
    return r;                                              // Rckgabewert
    }
    
// Quotient des gegebenen Monoms und eines weiteren Monoms:
// Es wird vorausgesetzt, dass der Koeffizient des ersten Monoms durch den des zweiten Monoms teilbar ist
// und dass die Exponenten im zweiten Monom hchstens so gro sind wie die im ersten. Trifft diese Voraussetzung nicht zu,
// so ist der Rckgabewert undefined.
// m ... Zweites Monom (Divisor)
  
  div (m) {
    var r = new Monomial();                                // Neues Monom (fr Quotient)
    if (this.coeff%m.coeff != 0n) return undefined;        // Rckgabewert, falls Koeffizientendivision nicht aufgeht
    r.coeff = this.coeff/m.coeff;                          // Quotient der Koeffizienten als Koeffizient des Ergebnis-Monoms
    for (var v=0; v<variables.length; v++) {               // Fr alle Variablen-Indizes ...
      var e = this.expo[v]-m.expo[v];                      // Differenz der Exponenten
      if (e < 0) return undefined;                         // Rckgabewert, falls Differenz negativ
      r.expo[v] = e;                                       // Differenz der gegebenen Exponenten als Exponent im Ergebnis-Monom
      }
    return r;                                              // Rckgabewert
    }
    
// Umwandlung in eine Zeichenkette (Provisorium fr Testzwecke):
// f ... Flag fr erstes Monom innerhalb eines Polynoms

  toString (f) {
    if (this.coeff == 0n) return (f ? "0" : "");           // Rckgabewert, falls Nullmonom
    var s = String(this.coeff);                            // Zeichenkette fr Koeffizient
    if (s.charAt(0) == "-") s = "- "+s.substring(1);       // Falls Minuszeichen, Leerzeichen einfgen
    else if (!f) s = "+ "+s;                               // Falls Pluszeichen nicht am Anfang, Pluszeichen und Leerzeichen einfgen 
    for (var i=0; i<variables.length; i++) {               // Fr alle Variablen-Indizes ...
      var e = this.expo[i];                                // Aktueller Exponent
      if (e != 0) {                                        // Falls Exponent ungleich 0 ...
        s += " "+variables.charAt(i);                      // Variable hinzufgen
        if (e != 1) s += "^"+e;                            // Falls Exponent grer als 1, '^' und Exponent hinzufgen
        }
      }
    return s;                                              // Rckgabewert
    }
    
// Hilfsroutine: Umwandlung des Koeffizienten (ungleich 0) in eine Zeichenkette
// f ... Flag fr erstes Monom innerhalb eines Polynoms; im Falle f == true werden Pluszeichen weggelassen 
// und Minuszeichen mit Zwischenraum vorangestellt; im Falle f == false wird auf jeden Fall ein Plus- oder Minuszeichen
// als Rechenzeichen vorangestellt, und zwar mit Zwischenraum.
  
  coeffToString (f) {
    var c = this.coeff;                                    // Abkrzung fr Koeffizient (Typ BigInt)
    var sign = (c>0 ? 1 : -1);                             // Vorzeichenfaktor
    var s = String(c);                                     // Koeffizient als Zeichenkette
    if (sign < 0) s = s.substring(1);                      // Gegebenenfalls Minuszeichen weglassen
    if (s == "1" && !this.isInt()) s = "";                 // Gegebenenfalls Koeffizient 1 weglassen
    if (f && sign < 0) s = "- "+s;                         // Gegebenenfalls Vorzeichen - mit Leerzeichen
    if (!f) {                                              // Falls nicht erstes Monom ...
      if (sign > 0) s = "+ "+s;                            // Rechenzeichen + und Betrag
      if (sign < 0) s = "- "+s;                            // Rechenzeichen - und Betrag
      } 
    return s;                                              // Rckgabewert 
    }
    
// Breite in Pixeln:
// f ... Flag fr erstes Monom innerhalb des Polynoms; im Falle f == true werden Pluszeichen weggelassen 
// und Minuszeichen mit Zwischenraum vorangestellt; im Falle f == false wird auf jeden Fall ein Plus- oder Minuszeichen
// als Rechenzeichen vorangestellt, und zwar mit Zwischenraum.
// Die Methode width sollte der Methode write entsprechen.

  width (f) {
    var w0 = widthPix(" ")/2;                              // Zwischenraum (Pixel)
    var s = this.coeffToString(f);                         // Zeichenkette fr Koeffizient (kann leer sein)
    var w = widthPix(s);                                   // Breite des Koeffizienten (Pixel)
    if (s != "" && !this.isInt()) w += w0;                 // Eventuell Zwischenraum zwischen Koeffizient und Variablen addieren
    for (var i=0; i<variables.length; i++) {               // Fr alle Variablen-Indizes ...
      var e = this.expo[i];                                // Zugehriger Exponent
      if (e == 0) continue;                                // Falls Exponent gleich 0, weiter zum nchsten Index
      w += widthPix(variables.charAt(i));                  // Breite der aktuellen Variablen addieren
      if (e == 1) {w += w0; continue;}                     // Falls Exponent gleich 1, weiter zum nchsten Index (Zwischenraum)
      w += widthPix(String(e));                            // Breite des Exponenten addieren
      w += w0;                                             // Breite des Zwischenraums addieren
      }
    if (!this.isInt()) w -= w0;                            // Falls Variable vorhanden, letzte Zwischenraum-Breite wieder subtrahieren   
    return w;                                              // Rckgabewert
    }
    
// Grafikausgabe:
// (x,y) ... Position (Pixel)
// f ....... Flag fr ersten Summanden
// Die vorkommenden Variablen werden alphabetisch geordnet.
// Rckgabewert: Position am Ende des Monoms (Pixel)

  write (x, y, f) {
    var w0 = widthPix(" ")/2;                              // Zwischenraum (Pixel)
    var s = this.coeffToString(f);                         // Zeichenkette fr Koeffizient (kann leer sein)
    x = writeString(s,x,y);                                // Zeichenkette ausgeben, neue Position
    if (s != "" && !this.isInt()) x += w0;                 // Eventuell Zwischenraum zwischen Koeffizient und Variablen
    for (var i=0; i<variables0.length; i++) {              // Fr alle Buchstaben-Indizes ...
      var ch = variables0.charAt(i);                       // Aktueller Buchstabe
      var v = variables.indexOf(ch);                       // Position in variables oder -1
      if (v < 0) continue;                                 // Falls Buchstabe nicht vorkommt, weiter zum nchsten
      var e = this.expo[v];                                // Zugehriger Exponent
      if (e == 0) continue;                                // Falls Exponent 0, weiter zum nchsten Buchstaben
      x = writeString(ch,x,y);                             // Variable ausgeben, neue Position
      if (e == 1) {x += w0; continue;}                     // Falls Exponent 1, neue Position und weiter zum nchsten Buchstaben
      x = writeString(""+e,x,y-8);                         // Exponent ausgeben
      x += w0;                                             // Neue Position
      } // Ende for (i)
    if (!this.isInt()) x -= w0;                            // Falls Variable vorhanden, letzte Positionsnderung rckgngig machen
    return x;                                              // Rckgabewert
    }
    
// Einsetzen einer Zahl fr eine Variable:
// v ... Index der Variablen (0 bis variables.length-1)
// n ... Zahl (Typ BigInt)

  subst (v, n) {
    var m = this.clone();                                  // Kopie des gegebenen Monoms
    m.coeff = m.coeff*n**BigInt(this.expo[v]);             // Koeffizient
    m.expo[v] = 0;                                         // Exponent bezglich der Variablen
    return m;                                              // Rckgabewert
    }

  } // Ende der Klasse Monomial
  
//-------------------------------------------------------------------------------------------------
  
// Konstruktor-Ersatz fr Testzwecke:
// s ... Zeichenkette

function newMonomial (s) {
  var neg = false;                                         // Flag fr Minuszeichen (gelscht)
  var ch = s.charAt(0);                                    // Erstes Zeichen
  if (ch == "-") neg = true;                               // Falls Minuszeichen, Flag setzen
  if (ch == "+" || ch == "-") s = s.substring(1);          // Gegebenenfalls Plus- oder Minuszeichen weglassen
  for (var i=0; i<s.length; i++)                           // Fr alle Zeichen-Indizes ...
    if (!like(s.charAt(i),"\\d")) break;                   // Falls keine Ziffer, for-Schleife abbrechen
  var m = new Monomial(s.substring(0,i));                  // Neues Monom (Koeffizient)
  if (i == 0) m.coeff = 1n;                                // Falls keine Ziffern am Anfang, Koeffizient 1
  if (neg) m.changeSign();                                 // Falls Minuszeichen, Monom umkehren
  s = s.substring(i);                                      // Zeichenkette nach dem Koeffizienten
  while (s.length > 0) {                                   // Solange Zeichenkette nicht leer ... 
    ch = s.charAt(0);                                      // Erstes Zeichen
    var v = variables.indexOf(ch);                         // Index der Variablen
    for (i=1; i<s.length; i++)                             // Fr alle Zeichen-Indizes ab 1 ...
      if (like(s.charAt(i),"[a-z]")) break;                // Falls Buchstabe, for-Schleife abbrechen
    m.expo[v] = 1;                                         // Exponent zunchst gleich 1
    if (i == 1) {s = s.substring(i); continue;}            // Falls kein Exponent, while-Schleife fortsetzen
    m.expo[v] = Number(s.substring(2,i));                  // Exponent anpassen
    s = s.substring(i);                                    // Rest der Zeichenkette
    } // Ende while
  return m;                                                // Rckgabewert
  }
  
