// Klasse RatVal (rationale Zahl)
// Attribute numerator, denominator
// 21.05.2020 - 01.07.2020

class RatVal {

// Konstruktor:
// n ... Zeichenkette Zhler (optional, Defaultwert "0")
// d ... Zeichenkette Nenner (optional, Defaultwert "1")
// Statt Zeichenketten sind wegen der automatischen Typumwandlung auch BigInt-Objekte mglich.

  constructor (n, d) {
    this.numerator = BigInt(n?n:"0");                      // Zhler (BigInt-Objekt)
    this.denominator = BigInt(d?d:"1");                    // Nenner (BigInt-Objekt)
    this.normal();                                         // Normalisierung (Krzen, Nenner positiv)
    }
    
// Normalisierung: Bruch vollstndig gekrzt, Nenner positiv
    
  normal () {
    var f = gcd(this.numerator,this.denominator);          // Grter gemeinsamer Teiler von Zhler und Nenner
    this.numerator = this.numerator/f;                     // Zhler durch ggT dividieren
    this.denominator = this.denominator/f;                 // Nenner durch ggT dividieren
    if (this.denominator < 0n) {                           // Falls Nenner negativ ...
      this.numerator = -this.numerator;                    // Vorzeichen Zhler umkehren
      this.denominator = -this.denominator;                // Vorzeichen Nenner umkehren
      }
    }
    
  } // Ende der Klasse RatVal
  
// Absolutbetrag eines BigInt-Objekts:
// n ... Gegebene Zahl (Typ BigInt)

function abs (n) {
  return (n<0n ? -n : n);                                  // Rckgabewert
  }
  
// Grter gemeinsamer Teiler von zwei BigInt-Objekten (euklidischer Algorithmus):
// n1, n2 ... Gegebene natrliche Zahlen

function gcd (n1, n2) {
  var a = n1, b = n2;                                      // Argumente bernehmen
  while (true) {                                           // Endlosschleife ...
    var c = a%b;                                           // Divisionsrest
    if (c == 0n) return b;                                 // Falls Rest gleich 0, Rckgabewert
    a = b; b = c;                                          // Operanden der nchsten Division
    }
  }
  
// Addition:
// q1, q2 ... Summanden (RatVal-Objekte)
  
function addRatVal (q1, q2) {
  if (q1 == undefined || q2 == undefined)                  // Falls mindestens ein Summand undefiniert ...
    return undefined;                                      // Ergebnis undefiniert
  var d = q1.denominator*q2.denominator;                   // Nenner
  var s1 = q1.numerator*q2.denominator;                    // Zhler 1. Summand
  var s2 = q2.numerator*q1.denominator;                    // Zhler 2. Summand
  var n = s1+s2;                                           // Zhler insgesamt
  return new RatVal(n,d);                                  // Summe (normalisiert) als Rckgabewert
  }
  
// Subtraktion:
// q1 ... Minuend (RatVal-Objekt)
// q2 ... Subtrahend (RatVal-Objekt)
  
function subRatVal (q1, q2) {
  if (q1 == undefined || q2 == undefined)                  // Falls Minuend oder Subtrahend undefiniert .. 
    return undefined;                                      // Ergebnis undefiniert
  var d = q1.denominator*q2.denominator;                   // Nenner 
  var s1 = q1.numerator*q2.denominator;                    // Zhler Minuend
  var s2 = q2.numerator*q1.denominator;                    // Zhler Subtrahend
  var n = s1-s2;                                           // Zhler insgesamt
  return new RatVal(n,d);                                  // Differenz (normalisiert) als Rckgabewert
  }
  
// Multiplikation:
// q1, q2 ... Faktoren (RatVal-Objekte)

function mulRatVal (q1, q2) {
  if (q1 == undefined || q2 == undefined)                  // Falls mindestens ein Faktor undefiniert ... 
    return undefined;                                      // Ergebnis undefiniert
  var n = q1.numerator*q2.numerator;                       // Zhler
  var d = q1.denominator*q2.denominator;                   // Nenner
  return new RatVal(n,d);                                  // Produkt (normalisiert) als Rckgabewert
  }
  
// Division:
// q1 ... Dividend (RatVal-Objekt)
// q2 ... Divisor (RatVal-Objekt)

function divRatVal (q1, q2) {
  if (q1 == undefined || q2 == undefined)                  // Falls Dividend oder Divisor undefiniert ...
    return undefined;                                      // Ergebnis undefiniert
  var n = q1.numerator*q2.denominator;                     // Zhler
  var d = q1.denominator*q2.numerator;                     // Nenner
  return new RatVal(n,d);                                  // Quotient (normalisiert) als Rckgabewert
  }
  
// Kehrwert:
// q ... Gegebene Zahl (RatVal-Objekt)

function recRatVal (q) {
  if (q == undefined) return undefined;                    // Falls Zahl undefiniert, Ergebnis undefiniert
  return new RatVal(q.denominator,q.numerator);            // Kehrwert (normalisiert) als Rckgabewert
  }
  
// berprfung der Ganzzahligkeit:

function isInteger (q) {
  return (q.numerator%q.denominator == 0n);                // Rckgabewert
  }
  
// Signum-Funktion:
// q ... Gegebene Zahl (RatVal-Objekt, normalisiert)

function signum (q) {
  if (q.numerator == 0n) return 0;                         // Rckgabewert, falls Zhler gleich 0
  return (q.numerator>0n ? 1 : -1);                        // Rckgabewert, falls Zhler ungleich 0
  }  
  
// Vergleich zweier Zahlen
// q1, q2 ... Gegebene Zahlen (RatVal-Objekte)
// Rckgabewert 0 fr q1 = q2, -1 fr q1 < q2, +1 fr q1 > q2

function compare (q1, q2) {
  return (signum(subRatVal(q1,q2)));                       // Rckgabewert
  }
  
// Potenzierung:
// q1 ... Basis (RatVal-Objekt)
// q2 ... Exponent (RatVal-Objekt, ganzzahlig)

function powRatVal (q1, q2) {
  if (q1 == undefined || q2 == undefined)                  // Falls Basis oder Exponent undefiniert ... 
    return undefined;                                      // Ergebnis undefiniert
  var pos = true;                                          // Flag fr positiven Exponenten
  var p = new RatVal("1");                                 // Variable fr Produkt, Startwert 1
  var e = q2.numerator/q2.denominator;                     // Exponent als BigInt-Objekt
  if (e  < 0n) {                                           // Falls Exponent negativ ...
      pos = false;                                         // Flag fr positiven Exponenten lschen 
      e = abs(e);                                          // Exponent durch Betrag ersetzen
      }
  while (e > 0n) {                                         // Solange e positiv ...
    p = mulRatVal(p,q1);                                   // Bisheriges Produkt mit Basis multiplizieren
    e = e-1n;                                              // e um 1 erniedrigen
    }
  if (!pos) p = recRatVal(p);                              // Falls Exponent negativ, Kehrwert bilden
  return p;                                                // Rckgabewert
  }
  
// Vorzeichenumkehr:
// q ... Gegebene Zahl (Typ RatVal)

function negateRatVal (q) {
  if (q == undefined) return undefined;                      // Falls Argument undefiniert, Ergebnis undefiniert
  return new RatVal(-q.numerator,q.denominator);             // Rckgabewert (Normalfall)
  }
  
// Ausgabe eines Bruchs (Zhler und Nenner gegeben):
// n ... Zhler (BigInt, positiv vorausgesetzt)
// d ... Nenner (BigInt, positiv vorausgesetzt)
// x ... Waagrechte Koordinate (Pixel)
// y ... Senkrechte Koordinate (Pixel)

function writeFrac2 (n, d, x, y) {
  var w1 = widthPix(n), w2 = widthPix(d);                  // Breite von Zhler und Nenner (Pixel)
  var w = Math.max(w1,w2);                                 // Breite des Bruchstrichs (Pixel)
  ctx.fillText(n,x+w-w1,y-7);                              // Zhler
  ctx.fillText(d,x+w-w2,y+7);                              // Nenner
  line(x,y-4,x+w,y-4);                                     // Bruchstrich
  }
  
// Ausgabe als Bruch (auch unecht):
// q ... Rationale Zahl (RatVal-Objekt, normalisiert)
// x ... Waagrechte Koordinate (Pixel)
// y ... Senkrechte Koordinate (Pixel)

function writeFrac (q, x, y) {
  if (!q) {                                                // Falls Bruch undefiniert ...
    ctx.fillText("?",x,y);                                 // Fragezeichen ausgeben 
    return;                                                // Abbrechen
    }  
  if (q.numerator < 0n) {                                  // Falls Bruch bzw. Zhler negativ ...
    ctx.fillText(symbolMinus,x,y);                         // Minuszeichen ausgeben
    x += widthPix(symbolMinus)+2;                          // Waagrechte Koordinate erhhen
    }
  var n = abs(q.numerator), d = q.denominator;             // Betrag des Zhlers, Nenner  
  if (d == 1n) ctx.fillText(n,x,y);                        // Entweder ganze Zahl ...
  else writeFrac2(n,d,x,y);                                // ... oder Bruch ausgeben
  }
  
// Ausgabe einer rationalen Zahl als Bruch oder gemischte Zahl:
// q ... Rationale Zahl (RatVal-Objekt, normalisiert)
// x ... Waagrechte Koordinate (Pixel)
// y ... Senkrechte Koordinate (Pixel)
 
function writeMix (q, x, y) {
  if (!q) {                                                // Falls Zahl undefiniert ...
    ctx.fillText("?",x,y);                                 // Fragezeichen ausgeben 
    return;                                                // Abbrechen
    } 
  if (q.numerator == 0n) {                                 // Falls Zahl gleich 0 ...
    ctx.fillText("0",x,y);                                 // Null ausgeben
    return;                                                // Abbrechen
    }      
  if (q.numerator < 0n) {                                  // Falls negative Zahl ...
    ctx.fillText(symbolMinus,x,y);                         // Minuszeichen ausgeben
    x += widthPix(symbolMinus)+2;                          // Waagrechte Koordinate erhhen
    }
  var n = abs(q.numerator), d = q.denominator;             // Betrag des Zhlers, Nenner
  var i = n/d;                                             // Ganzzahliger Anteil
  if (i > 0n) {                                            // Falls sinnvoll ... 
    ctx.fillText(i,x,y);                                   // Ganzzahligen Anteil ausgeben
    x += widthPix(i);                                      // Waagrechte Koordinate erhhen
    }
  var r = n%d;                                             // Zhler des Bruchs
  if (r > 0n) writeFrac2(r,d,x,y);                         // Falls sinnvoll, Bruch ausgeben
  }

// Vorbereitende Zerlegung fr Dezimalbruch-Schreibweise:
// q ... Gegebene rationale Zahl (RatVal-Objekt, normalisiert)
// Rckgabewert: Array von drei Zeichenketten; ganzzahliger Anteil (eventuell mit Minuszeichen), 
// normale Nachkommastellen (eventuell leere Zeichenkette), periodische Nachkommastellen (eventuell leere Zeichenkette)
  
function fragmentationDec (q) {
  if (q == undefined) return undefined;
  var a = new Array(3);                                    // Neues Array fr drei Zeichenketten
  var s0 = (q.numerator < 0n ? symbolMinus : "");          // Zeichenkette fr Ganze (eventuell mit Minuszeichen)
  var n = abs(q.numerator), d = abs(q.denominator);        // Betrge von Zhler und Nenner (Typ BigInt)  
  a[0] = s0+(n/d);                                         // Arrayelement mit Index 0                                            
  n = n%d;                                                 // Zhler des verbleibenden Bruchs  
  a[1] = a[2] = "";                                        // Weitere Arrayelemente
  if (n == 0n) return a;                                   // Falls ganze Zahl, abbrechen  
  var e2 = 0, e5 = 0;                                      // Exponenten der Faktoren 2 und 5 im Nenner, Startwerte 
  var h = d;                                               // Hilfsvariable h (Typ BigInt), zunchst gleich Nenner   
  while (h%2n == 0n) {                                     // Solange h durch 2 teilbar ...
    e2++; h = h/2n;                                        // Exponent erhhen, h durch 2 dividieren
    }
  while (h%5n == 0n) {                                     // Solange h durch 5 teilbar ...
    e5++; h = h/5n;                                        // Exponent erhhen, h durch 5 dividieren    
    }
  var e = Math.max(e2,e5);                                 // Maximum der beiden Exponenten    
  var s1 = "";                                             // Zeichenkette fr normale Nachkommastellen   
  for (var i=0; i<e; i++) {                                // Fr alle normalen Nachkommastellen ...
    n = n*10n;                                             // Zhler mit 10 multiplizieren
    s1 += n/d;                                             // Stelle hinzufgen
    n = n%d;                                               // Zhler aktualisieren
    }         
  a[1] = s1;                                               // Arrayelement mit Index 1
  if (n == 0n) return a;                                   // Falls endlicher Dezimalbruch, abbrechen  
  var s2 = "";                                             // Zeichenkette fr Periode
  var lp = 0;                                              // Variable fr Periodenlnge
  h = n;                                                   // Bisheriger Zhler
  do {                                                     // Wiederhole ...
    lp++;                                                  // Periodenlnge erhhen
    if (lp > 1000) {s2 += "..."; break;}                   // Eventuell Nothalt
    n = n*10n;                                             // Zhler mit 10 multiplizieren
    s2 += n/d;                                             // Stelle hinzufgen
    n = n%d;                                               // Zhler aktualisieren                      
    }
  while (n != h);                                          // ... solange Zhler verschieden vom frheren Zhler
  a[2] = s2;                                               // Arrayelement mit Index 2
  return a;                                                // Rckgabewert
  }
  
// Hilfsroutine: Zeichenkette mit Linie darber
// s ....... Zeichenkette
// (x,y) ... Position (Pixel)

function overline (s, x, y) {
  var w = widthPix(s);                                     // Breite (Pixel)  
  ctx.fillText(s,x,y);                                     // Zeichenkette ausgeben
  line(x,y-11,x+w,y-11);                                   // Linie zeichnen
  }
    
// Ausgabe als Dezimalbruch:
// q ....... Gegebene rationale Zahl (RatVal-Objekt) oder undefined
// (x,y) ... Position (Pixel)
  
function writeDec (q, x, y) {
  if (q == undefined) {                                    // Falls Zahl undefiniert ... 
    ctx.fillText("?",x,y);                                 // Fragezeichen
    return;                                                // Abbrechen
    }
  var a = fragmentationDec(q);                             // Array der drei Teilzeichenketten
  ctx.fillText(a[0],x,y);                                  // Ganzzahliger Anteil (eventuell mit Minuszeichen)
  if (a[1]+a[2] == "") return;                             // Falls keine Nachkommastellen, abbrechen
  x += widthPix(a[0]);                                     // Waagrechte Koordinate erhhen
  ctx.fillText(decimalSeparator,x,y);                      // Dezimaltrennzeichen (Komma oder Punkt)
  x += widthPix(decimalSeparator);                         // Waagrechte Koordinate erhhen
  ctx.fillText(a[1],x,y);                                  // Normale Nachkommastellen
  x += widthPix(a[1]);                                     // Waagrechte Koordinate erhhen
  overline(a[2],x,y);                                      // Periodische Nachkommastellen (mit Strich darber)
  }
  